/**
  ******************************************************************************
  * @file    cp_frame.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   ISO layer 2 framing code for the communication protocol
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

#include "project.h"

#include "cp.h"

/** @brief Handles an incoming packet from an established peer connection.
 *
 *  @param p_DataBuf Pointer to data packet
 *  @param p_PeerPort Packet length
 *  @param p_ProtInFSM communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * called in the context of a receiver task. 
 */

CP_STATUSCODE CP_DispatchInPacketToUpperLayer(unsigned char *p_DataBuf,unsigned long p_DataLen,PCP_PROCESSOR p_ProtInFSM)
{
    // the m_Queue member of p_ProtInFSM is the queue to post a response to.
    // You may answer packets inline (CP_FrameAndEmitPacket is serialzed among tasks via a mutex).
    unsigned short a_TagBitMask;
    CP_DYNTAGSTRUCT *a_DynTagStruct;
    CP_STATUSCODE a_Return = CP_STATUSCODE_CORRUPT; // default
    if (cp_SyntaxCheck(p_DataBuf,p_DataLen,&a_DynTagStruct,&a_TagBitMask) == CP_STATUSCODE_SUCCESS)
    {
        a_Return = CP_STATUSCODE_SUCCESS;   // new default
        // 1. determine whether we are authenticated (a client always starts out authenticated); if not, silently discard everything
        // but authentication packets.
        if ((CP_PeerIsAuthenticated(p_ProtInFSM) == CP_STATUSCODE_AUTHFAIL) && TEST_BIT_IN_BITMASK(a_TagBitMask,TAG_BITMASK_AUTH_PRESENT))
            cp_TestForAuthPacket(a_DynTagStruct,p_ProtInFSM); // will implicitly set the processor state to authenticated if successful authentication.
        if (CP_PeerIsAuthenticated(p_ProtInFSM) == CP_STATUSCODE_SUCCESS)   // no else due to the side effect!
        {
            if (TEST_BIT_IN_BITMASK(a_TagBitMask,TAG_BITMASK_RESPONSE_PRESENT))
            {
                CP_ProcessResponsePacket(a_DynTagStruct,p_ProtInFSM);
            }
            // this logic does not really exclude the possibility that both a payload and a response packet are tucked into the same TLV (which is forbidded by the spec).
            // Currently we prioritize the response and ignore payload packets in the same telegram. Should really reject such a packet during cp_SyntaxCheck().
            else
            {
                unsigned long a_OriginSeqNo;
                CP_STATUSCODE a_EvalResponse = cp_DissectIncomingPacket(a_DynTagStruct,p_ProtInFSM,&a_OriginSeqNo); 
                a_Return = cp_RespondPacket(a_OriginSeqNo,(a_EvalResponse == CP_STATUSCODE_SUCCESS)?CP_RESPONSE_ACK:CP_RESPONSE_GENERIC_NAK,p_ProtInFSM);
            }
        }
        cp_DeleteTagChain(&a_DynTagStruct);            
    };
    return a_Return;
}

/** @brief Reinitializes the layer 2 processing FSM (called when a new incoming packet starts).
 *
 *  @param p_ProtInFSM FSM control structure 
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 */

static CP_STATUSCODE CP_InitFSM(PCP_FSM_IN p_ProtInFSM)
{
    p_ProtInFSM->m_Flags = 0; 
    p_ProtInFSM->m_PendingReadLen = p_ProtInFSM->m_CharsRead = 0;
    CP_FSM_STATE_CHANGE(p_ProtInFSM,CP_FSM_IN_SCANNING);
    return CP_STATUSCODE_SUCCESS;  
}

/** @brief Resets the packet reader's input buffer.
 *
 *  @param p_ProtInFSM FSM control structure 
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * If the protocol requires dynamic buffer allocation for a new packet, here's the place to 
 * allocate the buffer.
 */

CP_STATUSCODE CP_PrepareInBuf(PCP_FSM_IN p_ProtInFSM)
{
    p_ProtInFSM->m_InPtr = p_ProtInFSM->m_ShortBuf;
    return CP_STATUSCODE_SUCCESS;
}

/** @brief Resets the packet reader's input buffer.
 *
 *  @param p_ProtInFSM FSM control structure 
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * If the protocol requires dynamic buffer allocation for a new packet, here's the place to 
 * reclaim the buffer.
 */

CP_STATUSCODE CP_ReleaseInBuf(PCP_FSM_IN p_ProtInFSM)
{
    return CP_STATUSCODE_SUCCESS;
}

/** @brief forwards the protocol parser fsm with a newly arrived character.
 *
 *  @param p_Char next character from input stream
 *  @param p_ProtInFSM communication processor
 *
 *  @return none (side effect on FSM) 
 */
/*
The parser for incoming packets theoretically works strictly on the ISO/OSI framing level 2. For this protocol, this means that the framing is decoded - the framing character is 
being stripped, the length is read and transpositions are reversed. However, in practice (in particular with embedded systems) we need to compromise and dilute the strict 
separation at times.

In this protocol implementation, the parser will reject packets with the length word specifiying

1. a length smaller than the minimum of one single TLV (2 words ~ 8 bytes)

or 

2. a length greater than the maximum receive unit (MRU) currently hardcoded into the firmware.

The background is that embedded controllers are ressource limited and can not progress arbitrarily long data; also, port scanners or malware may attempt to inject 
random data into a connection stream which may crash a fault intolerant application. A more sophisticated architecure may negotiate the MRU during protocol startup 
and/or dynamically allocate memory for incoming packets on demand.

A difficult question concerns the reaction to malformed packets. This implementation silently discards packets of inappropriate length. If a peer doesn't know about the MRU 
requirements (eg in an incompletly specified protocol), it will attempt to resend the packet forever, thus causing a queue congestion. We could catch this scenario 
by defining a specific response code CP_STATUSCODE_INVALID_LENGTH_RECEIVED and pass that one back to the peer. However, it is not straightforward to decide what a peer should 
do upon receiving this response, in particular because there is a possibility (not unlikely in protocol implementations over serial interfaces and remotely likely in faulty 
network stack implementations) that a well formed packet (meaning one with a correct length) has been corrupted during transmission; thus, a peer discarding a packet upon 
reception of this status code may erratically remove a good packet from its outbound queue. Typical solutions for this scenario include a high level timeout or a maximum 
number of transmission attempts before discarding a packet which however compromises the determinstic behavior of the communication. The primary guiding line here should be 
the liveness of the protocol, meaning that there should never be a scenario in which the packet flow stalls for an indeterministic amount of time.


  
*/ 
 

void CP_ProcessInChar(unsigned char p_Char,PCP_PROCESSOR p_Processor)
{
    PCP_FSM_IN a_Processor = &p_Processor->m_FSMStruct;
    switch (a_Processor->m_FSM)
    {
        case CP_FSM_IN_SCANNING:
            if (p_Char == CP_FRAMESTART)
                CP_FSM_STATE_CHANGE(a_Processor,CP_FSM_IN_FRAME_START_FOUND);
            break;
        case CP_FSM_IN_FRAME_START_FOUND:
            CP_CHECKFORTRANSPOSITION(p_Char,a_Processor, \
                                        { \
                                            a_Processor->m_PendingReadLen += (p_Char << 24); \
                                            CP_FSM_STATE_CHANGE(a_Processor,CP_FSM_IN_READ_LEN1); \
                                        } \
                                    )
            break;
        case CP_FSM_IN_READ_LEN1:
            CP_CHECKFORTRANSPOSITION(p_Char,a_Processor, \
                                        { \
                                            a_Processor->m_PendingReadLen += (p_Char << 16); \
                                            CP_FSM_STATE_CHANGE(a_Processor,CP_FSM_IN_READ_LEN2); \
                                        } \
                                    )
            break;
        case CP_FSM_IN_READ_LEN2:
            CP_CHECKFORTRANSPOSITION(p_Char,a_Processor, \
                                        { \
                                            a_Processor->m_PendingReadLen += (p_Char << 8); \
                                            CP_FSM_STATE_CHANGE(a_Processor,CP_FSM_IN_READ_LEN3); \
                                        } \
                                    )          
            break;
        case CP_FSM_IN_READ_LEN3:
            CP_CHECKFORTRANSPOSITION(p_Char,a_Processor, \
                                        { \
                                            a_Processor->m_PendingReadLen += (unsigned long)p_Char; \
                                            if (a_Processor->m_PendingReadLen < 2*(sizeof(unsigned long))) \
                                                goto ResetFSM; \
                                            if (a_Processor->m_PendingReadLen > sizeof(a_Processor->m_ShortBuf)) \
                                            { \
                                                goto ResetFSM; \
                                            } \
                                            if (!a_Processor->m_PendingReadLen) \
                                            { \
                                                CP_DispatchInPacketToUpperLayer(a_Processor->m_InPtr,a_Processor->m_CharsRead,p_Processor); \
                                                CP_ReleaseInBuf(a_Processor); \
                                                goto ResetFSM; \
                                            } \
                                            CP_PrepareInBuf(a_Processor); \
                                            CP_FSM_STATE_CHANGE(a_Processor,CP_FSM_IN_READING_PAYLOAD); \
                                        } \
                                    )          
            break;
        case CP_FSM_IN_READING_PAYLOAD:
            CP_CHECKFORTRANSPOSITION(p_Char,a_Processor, \
                                        { \
                                            a_Processor->m_InPtr[a_Processor->m_CharsRead] = p_Char; \
                                            if (++(a_Processor->m_CharsRead) >= a_Processor->m_PendingReadLen) \
                                            { \
                                                CP_DispatchInPacketToUpperLayer(a_Processor->m_InPtr,a_Processor->m_CharsRead,p_Processor); \
                                                CP_ReleaseInBuf(a_Processor);
                                                goto ResetFSM; \
                                            } \
                                        } \
                                    )
            break;
        case CP_FSM_IN_ABORT:
ResetFSM:
            CP_InitFSM(a_Processor);
    }
}

/** @brief outputs a single character, transposes it if necessary.
 *
 *  @param p_Char Next untransposed character to pass to output stream
 *  @param p_ProtInFSM communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 */

CP_STATUSCODE CP_EmitCharTranspose(unsigned char p_Char,PCP_PROCESSOR p_Processor)
{
    switch (p_Char)
    {
        case CP_FRAMESTART: case CP_ESCAPE:
            if (CP_EmitCharVerbatim(CP_ESCAPE,p_Processor) == CP_STATUSCODE_SUCCESS)
                return CP_EmitCharVerbatim(TRANSPOSE(p_Char),p_Processor);
            else
                return CP_STATUSCODE_CONNLOSS;
            break;
        default:
            return CP_EmitCharVerbatim(p_Char,p_Processor);
    }
}

/** @brief frames a data packet according to the layer 2 specification, transposes individual characters if neccesary
 *
 *  @param p_Packet raw packet to send
 *  @param p_Packet packet length
 *  @param p_ProtInFSM communication processor
 *
 *  @return status code (see CP_STATUSCODE_XXX enumerator in cp_frame.h)
 *
 * Called in the context of a client or server task and a receiver task. Function is reentrant and serialized.
 */

CP_STATUSCODE CP_FrameAndEmitPacket(unsigned char *p_Packet,unsigned long p_PacketLen,PCP_PROCESSOR p_Processor)
{
    CP_STATUSCODE a_Result;
    unsigned char *a_Current,*a_End;
    xSemaphoreTake(p_Processor->m_WriterMutex,portMAX_DELAY);
    a_Current = p_Packet;
    a_End = &p_Packet[p_PacketLen]; 
    a_Result = CP_EmitCharVerbatim(CP_FRAMESTART,p_Processor);
    a_Result = CP_EmitCharTranspose(p_PacketLen>>24,p_Processor);
    a_Result = CP_EmitCharTranspose(p_PacketLen>>16,p_Processor);
    a_Result = CP_EmitCharTranspose(p_PacketLen>>8,p_Processor);
    a_Result = CP_EmitCharTranspose(p_PacketLen & 0xff,p_Processor);
    
    while ((a_Result == CP_STATUSCODE_SUCCESS) && (a_Current < a_End))
    {
        unsigned long a_ChunkLen = 0;
        while ((&a_Current[a_ChunkLen] < a_End) && (a_Current[a_ChunkLen] != CP_ESCAPE) && (a_Current[a_ChunkLen] != CP_FRAMESTART))
            a_ChunkLen++;
        if (a_ChunkLen)
        {
            a_Result = CP_EmitChunkVerbatim(a_Current,a_ChunkLen,p_Processor);
            a_Current += a_ChunkLen;
        }
        if (a_Current < a_End)
        {
            a_Result = CP_EmitCharVerbatim(CP_ESCAPE,p_Processor);
            if (a_Result == CP_STATUSCODE_SUCCESS)
                a_Result = CP_EmitCharVerbatim(TRANSPOSE(*a_Current),p_Processor);
            a_Current++;
        }
    }
    xSemaphoreGive(p_Processor->m_WriterMutex);
    return a_Result;
}

/** @brief obtains the next sequence number for a packet to send
 *
 *  @param p_CommHandle communication processor
 *
 *  @return next following sequence number
 *
 * Note: The Specification does NOT require sequential Telegram Ids. In order to make sure that nothing relies on sequential Ids, we provide two implementations
 * which must both pass all regression tests. It is perfectly legal for different sides of a communication to use different schemes, so in cp.h, we couple the
 * server and client functionality with the different strategies.
 */

unsigned long CP_NextSeqNo(CP_COMMHANDLE p_CommHandle)
{
#ifdef TEST_SEQUENTIAL_NUMBERS
    return (p_CommHandle->m_CurrentSeqNr++);
#else
    unsigned long aNextProbedNo;
    do
    {
        aNextProbedNo = rand();
    }   while (aNextProbedNo == p_CommHandle->m_CurrentSeqNr); // depending on the quality of the random generator, this may or may not happen ever
    p_CommHandle->m_CurrentSeqNr = aNextProbedNo;
    return aNextProbedNo;
#endif
}
